<?php

if ( ! function_exists( 'warren_core_include_course_shortcodes' ) ) {
	/**
	 * Function that includes shortcodes
	 */
	function warren_core_include_course_shortcodes() {
		foreach ( glob( WARREN_CORE_PLUGINS_PATH . '/lms/post-types/course/shortcodes/*/include.php' ) as $shortcode ) {
			include_once $shortcode;
		}
	}

	add_action( 'qode_framework_action_before_shortcodes_register', 'warren_core_include_course_shortcodes' );
}

if ( ! function_exists( 'warren_core_include_course_widgets' ) ) {
	/**
	 * Function that includes widgets
	 */
	function warren_core_include_course_widgets() {
		foreach ( glob( WARREN_CORE_PLUGINS_PATH . '/lms/post-types/course/widgets/*/include.php' ) as $widget ) {
			include_once $widget;
		}
	}

	add_action( 'qode_framework_action_before_widgets_register', 'warren_core_include_course_widgets' );
}

if ( ! function_exists( 'warren_core_generate_course_single_layout' ) ) {
	function warren_core_generate_course_single_layout() {
		$template = 'default';

		return $template;
	}

	add_filter( 'warren_core_filter_course_single_layout', 'warren_core_generate_course_single_layout' );
}

if ( ! function_exists( 'warren_core_add_administrator_capabilites_course' ) ) {
	function warren_core_add_administrator_capabilites_course( $cpts ) {
		$cpts[] = 'course';

		return $cpts;
	}

	add_filter( 'warren_core_filter_administrator_cpts', 'warren_core_add_administrator_capabilites_course' );
}

if ( ! function_exists( 'warren_core_course_set_admin_options_map_position' ) ) {
	/**
	 * Function that set dashboard admin options map position for this module
	 *
	 * @param int $position
	 * @param string $map
	 *
	 * @return int
	 */
	function warren_core_course_set_admin_options_map_position( $position, $map ) {

		if ( 'course' === $map ) {
			$position = 60;
		}

		return $position;
	}

	add_filter( 'warren_core_filter_admin_options_map_position', 'warren_core_course_set_admin_options_map_position', 10, 2 );
}

if ( ! function_exists( 'warren_core_enqueue_course_assets' ) ) {
	/**
	 * Function that enqueue 3rd party plugins script
	 */
	function warren_core_enqueue_course_assets() {

		if ( is_singular( 'course' ) ) {
			wp_enqueue_style( 'perfect-scrollbar', WARREN_CORE_URL_PATH . 'assets/plugins/perfect-scrollbar/perfect-scrollbar.css', array() );
			wp_enqueue_script( 'perfect-scrollbar', WARREN_CORE_URL_PATH . 'assets/plugins/perfect-scrollbar/perfect-scrollbar.jquery.min.js', array( 'jquery' ), false, true );
		}
	}

	add_action( 'warren_core_action_before_main_css', 'warren_core_enqueue_course_assets' );
}

if ( ! function_exists( 'warren_core_include_course_single_scripts' ) ) {
	/**
	 * Enqueue 3rd party scripts for current module single page
	 */
	function warren_core_include_course_single_scripts() {

		if ( is_singular( 'course' ) && is_author()) {
			// Include tabs jquery
			wp_enqueue_script( 'jquery-ui-tabs' );
		}
	}

	add_action( 'warren_core_action_before_main_js', 'warren_core_include_course_single_scripts' );
}

if ( ! function_exists( 'warren_core_set_page_title_for_course_single' ) ) {
	/**
	 * Function that disable page title area
	 *
	 * @param bool $enable_page_title
	 *
	 * @return bool
	 */
	function warren_core_set_page_title_for_course_single( $enable_page_title ) {

		if ( is_singular( 'course' ) ) {
			$option = 'no' !== warren_core_get_post_value_through_levels( 'qodef_course_single_enable_page_title' );

			if ( isset( $option ) ) {
				$enable_page_title = $option;
			}

			$meta_option = get_post_meta( get_the_ID(), 'qodef_enable_page_title', true );

			if ( ! empty( $meta_option ) ) {
				$enable_page_title = $meta_option;
			}
		}

		return $enable_page_title;
	}

	add_filter( 'warren_filter_enable_page_title', 'warren_core_set_page_title_for_course_single' );
}

if ( ! function_exists( 'warren_core_set_course_single_sidebar_grid_gutter_classes' ) ) {
	/**
	 * Function that returns grid gutter classes
	 *
	 * @param string $classes
	 *
	 * @return string
	 */
	function warren_core_set_course_single_sidebar_grid_gutter_classes( $classes ) {

		if ( is_singular( 'course' ) ) {
			$option = warren_core_get_post_value_through_levels( 'qodef_course_single_sidebar_grid_gutter' );

			if ( ! empty( $option ) ) {
				$classes = 'qodef-gutter--' . esc_attr( $option );
			}

			$meta_option = get_post_meta( get_the_ID(), 'qodef_page_sidebar_grid_gutter', true );

			if ( ! empty( $meta_option ) ) {
				$classes = 'qodef-gutter--' . esc_attr( $meta_option );
			}
		}

		return $classes;
	}

	add_filter( 'warren_filter_grid_gutter_classes', 'warren_core_set_course_single_sidebar_grid_gutter_classes' );
}

if ( ! function_exists( 'warren_core_set_course_single_sidebar_grid_gutter_styles' ) ) {
	/**
	 * Function that returns grid gutter styles
	 *
	 * @param array $styles
	 *
	 * @return array
	 */
	function warren_core_set_course_single_sidebar_grid_gutter_styles( $styles ) {

		if ( is_singular( 'course' ) ) {
			$styles = warren_core_get_gutter_custom_styles( 'qodef_course_single_sidebar_grid_gutter_', 'qodef_page_sidebar_grid_gutter_' );
		}

		return $styles;
	}

	add_filter( 'warren_filter_grid_gutter_styles', 'warren_core_set_course_single_sidebar_grid_gutter_styles' );
}

if ( ! function_exists( 'warren_core_set_course_single_sidebar_layout' ) ) {
	/**
	 * Function that return sidebar layout
	 *
	 * @param string $layout
	 *
	 * @return string
	 */
	function warren_core_set_course_single_sidebar_layout( $layout ) {

		if ( is_singular( 'course' ) ) {
			$option = warren_core_get_post_value_through_levels( 'qodef_course_single_sidebar_layout' );

			if ( ! empty( $option ) ) {
				$layout = $option;
			}

			$meta_option = get_post_meta( get_the_ID(), 'qodef_page_sidebar_layout', true );

			if ( ! empty( $meta_option ) ) {
				$layout = $meta_option;
			}
		}

		return $layout;
	}

	add_filter( 'warren_filter_sidebar_layout', 'warren_core_set_course_single_sidebar_layout' );
}

if ( ! function_exists( 'warren_core_set_course_single_custom_sidebar_name' ) ) {
	/**
	 * Function that return sidebar name
	 *
	 * @param string $sidebar_name
	 *
	 * @return string
	 */
	function warren_core_set_course_single_custom_sidebar_name( $sidebar_name ) {

		if ( is_singular( 'course' ) ) {
			$option = warren_core_get_post_value_through_levels( 'qodef_course_single_custom_sidebar' );

			if ( ! empty( $option ) ) {
				$sidebar_name = $option;
			}

			$meta_option = get_post_meta( get_the_ID(), 'qodef_page_custom_sidebar', true );

			if ( ! empty( $meta_option ) ) {
				$sidebar_name = $meta_option;
			}
		}

		return $sidebar_name;
	}

	add_filter( 'warren_filter_sidebar_name', 'warren_core_set_course_single_custom_sidebar_name' );
}

if ( ! function_exists( 'warren_core_course_breadcrumbs_title' ) ) {
	/**
	 * Improve main breadcrumb template with additional cases
	 *
	 * @param string|html $wrap_child
	 * @param array $settings
	 *
	 * @return string|html
	 */
	function warren_core_course_breadcrumbs_title( $wrap_child, $settings ) {
		if ( is_tax( 'course-category' ) ) {
			$wrap_child  = '';
			$term_object = get_term( get_queried_object_id(), 'course-category' );

			if ( isset( $term_object->parent ) && 0 !== $term_object->parent ) {
				$parent      = get_term( $term_object->parent );
				$wrap_child .= sprintf( $settings['link'], get_term_link( $parent->term_id ), $parent->name ) . $settings['separator'];
			}

			$wrap_child .= sprintf( $settings['current_item'], single_cat_title( '', false ) );
		} elseif ( is_tax( 'course-tag' ) ) {
			$wrap_child  = '';
			$term_object = get_term( get_queried_object_id(), 'course-tag' );

			if ( isset( $term_object->parent ) && 0 !== $term_object->parent ) {
				$parent      = get_term( $term_object->parent );
				$wrap_child .= sprintf( $settings['link'], get_term_link( $parent->term_id ), $parent->name ) . $settings['separator'];
			}

			$wrap_child .= sprintf( $settings['current_item'], single_cat_title( '', false ) );
		} elseif ( is_singular( 'course' ) ) {
			$wrap_child = '';
			$post_terms = wp_get_post_terms( get_the_ID(), 'course-category' );

			if ( ! empty( $post_terms ) ) {
				$post_term = $post_terms[0];
				if ( isset( $post_term->parent ) && 0 !== $post_term->parent ) {
					$parent      = get_term( $post_term->parent );
					$wrap_child .= sprintf( $settings['link'], get_term_link( $parent->term_id ), $parent->name ) . $settings['separator'];
				}
				$wrap_child .= sprintf( $settings['link'], get_term_link( $post_term ), $post_term->name ) . $settings['separator'];
			}

			$wrap_child .= sprintf( $settings['current_item'], get_the_title() );
		}

		return $wrap_child;
	}

	add_filter( 'warren_core_filter_breadcrumbs_content', 'warren_core_course_breadcrumbs_title', 10, 2 );
}

if ( ! function_exists( 'warren_core_get_course_params' ) ) {
	function warren_core_get_course_params( $id = '' ) {
		$course_id = ! empty( $id ) ? $id : get_the_ID();
		$params    = array();

		$show_members = warren_core_get_post_value_through_levels( 'qodef_course_single_show_members', $course_id );

		if ( 'yes' === $show_members ) {
			$params['show_members'] = true;
		} else {
			$params['show_members'] = false;
		}

		return $params;
	}
}

if ( ! function_exists( 'warren_core_generate_course_archive_with_shortcode' ) ) {
	/**
	 * Function that executes team list shortcode with params on archive pages
	 *
	 * @param string $tax - type of taxonomy
	 * @param string $tax_slug - slug of taxonomy
	 * @param array $search_params
	 */
	function warren_core_generate_course_archive_with_shortcode( $tax = '', $tax_slug = '', $search_params = array() ) {
		$params = array();

		$params['layout']                     = warren_core_get_post_value_through_levels( 'qodef_course_archive_item_layout' );
		$params['behavior']                   = 'columns';
		$params['orderby']                    = 'course-featured';
		$params['columns']                    = warren_core_get_post_value_through_levels( 'qodef_course_archive_columns' );
		$params['space']                      = warren_core_get_post_value_through_levels( 'qodef_course_archive_space' );
		$params['space_custom']               = warren_core_get_post_value_through_levels( 'qodef_course_archive_space_custom' );
		$params['space_custom_1512']          = warren_core_get_post_value_through_levels( 'qodef_course_archive_space_custom_1512' );
		$params['space_custom_1200']          = warren_core_get_post_value_through_levels( 'qodef_course_archive_space_custom_1200' );
		$params['space_custom_880']           = warren_core_get_post_value_through_levels( 'qodef_course_archive_space_custom_880' );
		$params['vertical_space']             = warren_core_get_post_value_through_levels( 'qodef_course_archive_vertical_space' );
		$params['vertical_space_custom']      = warren_core_get_post_value_through_levels( 'qodef_course_archive_vertical_space_custom' );
		$params['vertical_space_custom_1512'] = warren_core_get_post_value_through_levels( 'qodef_course_archive_vertical_space_custom_1512' );
		$params['vertical_space_custom_1200'] = warren_core_get_post_value_through_levels( 'qodef_course_archive_vertical_space_custom_1200' );
		$params['vertical_space_custom_880']  = warren_core_get_post_value_through_levels( 'qodef_course_archive_vertical_space_custom_880' );
		$params['columns_responsive']         = warren_core_get_post_value_through_levels( 'qodef_course_archive_columns_responsive' );
		$params['columns_1512']               = warren_core_get_post_value_through_levels( 'qodef_course_archive_columns_1512' );
		$params['columns_1368']               = warren_core_get_post_value_through_levels( 'qodef_course_archive_columns_1368' );
		$params['columns_1200']               = warren_core_get_post_value_through_levels( 'qodef_course_archive_columns_1200' );
		$params['columns_1024']                = warren_core_get_post_value_through_levels( 'qodef_course_archive_columns_1024' );
		$params['columns_880']                = warren_core_get_post_value_through_levels( 'qodef_course_archive_columns_880' );
		$params['columns_680']                = warren_core_get_post_value_through_levels( 'qodef_course_archive_columns_680' );
		$params['pagination_type']            = warren_core_get_post_value_through_levels( 'qodef_course_archive_pagination_type' );

		$params['additional_params'] = 'tax';
		$params['tax']               = $tax;
		$params['tax_slug']          = $tax_slug;

		if ( ! empty( $search_params ) && is_array( $search_params ) ) {
			foreach ( $search_params as $key => $value ) {

				if ( 'all' === $value ) {
					continue;
				}

				switch ( $key ) {
					case 'category':
						$params['additional_params'] = 'tax';
						$params['tax']               = 'course-' . $key;
						$params['tax_slug']          = $value;
						break;
					case 'instructor':
						$params['additional_instructor'] = 'instructor';
						$params['instructor']            = $value;
						break;
					case 'price':
						$params['price'] = $value;
						break;
				}
			}
		}

		echo WarrenCore_Course_List_Shortcode::call_shortcode( $params );
	}
}

if ( ! function_exists( 'warren_core_course_curriculum_info' ) ) {
	/**
	 * Function that returns information about how many lessons and quizzes are in section
	 *
	 * @param array $sections - array of sections in one course
	 *
	 * @return array $params
	 */
	function warren_core_course_curriculum_info( $sections ) {
		$params['lesson'] = array(
			'count' => 0,
			'label' => esc_html__( 'Lessons: ', 'warren-core' ),
		);
		$params['quiz']   = array(
			'count' => 0,
			'label' => esc_html__( 'Quizzes: ', 'warren-core' ),
		);

		if ( count( $sections ) ) {
			foreach ( $sections as $section ) {
				$subsections = $section['qodef_course_curriculum_subsections'];
				if ( count( $subsections ) ) {
					foreach ( $subsections as $subsection ) {
						$params[ $subsection['qodef_course_subsection_item'] ]['count'] ++;
					}
				}
			}
		}

		if ( 1 === $params['lesson']['count'] ) {
			$params['lesson']['label'] = esc_html__( 'Lesson: ', 'warren-core' );
		}

		if ( 1 === $params['quiz']['count'] ) {
			$params['quiz']['label'] = esc_html__( 'Quiz: ', 'warren-core' );
		}

		return $params;
	}
}

if ( ! function_exists( 'warren_core_course_section_info_count' ) ) {
	/**
	 * Function that returns information about how many of each lesson types are in section
	 *
	 * @param array $subsections - array of subsections in one course
	 *
	 * @return array $params
	 */
	function warren_core_course_section_info_count( $subsections ) {
		$params['video']   = array(
			'count' => 0,
			'label' => esc_html__( 'Videos', 'warren-core' ),
		);
		$params['audio']   = array(
			'count' => 0,
			'label' => esc_html__( 'Audios', 'warren-core' ),
		);
		$params['reading'] = array(
			'count' => 0,
			'label' => esc_html__( 'Readings', 'warren-core' ),
		);

		if ( count( $subsections ) ) {
			foreach ( $subsections as $subsection ) {
				if ( 'lesson' === $subsection['qodef_course_subsection_item'] ) {
					$id   = $subsection['qodef_course_subsection_item_lesson'];
					$type = get_post_meta( $id, 'qodef_lesson_type', true );

					switch ( $type ) {
						case 'video':
							$params['video']['count'] ++;
							break;
						case 'audio':
							$params['audio']['count'] ++;
							break;
						case 'reading':
						default:
							$params['reading']['count'] ++;
							break;
					}
				}
			}
		}

		if ( 1 === $params['reading']['count'] ) {
			$params['reading']['label'] = esc_html__( 'Reading', 'warren-core' );
		}

		if ( 1 === $params['video']['count'] ) {
			$params['video']['label'] = esc_html__( 'Video', 'warren-core' );
		}

		if ( 1 === $params['audio']['count'] ) {
			$params['audio']['label'] = esc_html__( 'Audio', 'warren-core' );
		}

		return $params;
	}
}

if ( ! function_exists( 'warren_core_lms_get_duration_label' ) ) {
	/**
	 * Function that returns duration label parameter in dependency of duration and duration unit
	 *
	 * @param string $duration - duration number
	 * @param string $duration_unit - duration unit
	 *
	 * @return string $duration_label
	 */
	function warren_core_lms_get_duration_label( $duration, $duration_unit ) {
		$duration       = intval( $duration );
		$duration_label = '';

		switch ( $duration_unit ) {
			case 'minutes':
				if ( $duration > 1 ) {
					$duration_label = esc_html__( 'minutes', 'warren-core' );
				} else {
					$duration_label = esc_html__( 'minute', 'warren-core' );
				}
				break;
			case 'hours':
				if ( $duration > 1 ) {
					$duration_label = esc_html__( 'hours', 'warren-core' );
				} else {
					$duration_label = esc_html__( 'hour', 'warren-core' );
				}
				break;
			case 'days':
				if ( $duration > 1 ) {
					$duration_label = esc_html__( 'days', 'warren-core' );
				} else {
					$duration_label = esc_html__( 'day', 'warren-core' );
				}
				break;
			case 'weeks':
				if ( $duration > 1 ) {
					$duration_label = esc_html__( 'weeks', 'warren-core' );
				} else {
					$duration_label = esc_html__( 'week', 'warren-core' );
				}
				break;
		}

		return $duration_label;
	}
}

if ( ! function_exists( 'warren_core_course_get_instructors' ) ) {
	/**
	 * Function that returns instructors list
	 *
	 * @return array $instructors
	 */
	function warren_core_course_get_instructors() {
		$instructors_query = get_users(
			array(
				'role__in' => array(
					'instructor',
				),
				'fields'   => array(
					'ID',
					'display_name',
				),
			)
		);

		$instructors = array();

		foreach ( $instructors_query as $instructor ) {
			$instructors[ $instructor->ID ] = $instructor->display_name;
		}

		return $instructors;
	}
}

if ( ! function_exists( 'warren_core_course_save_final_price' ) ) {
	/**
	 * Function that adds final price meta for course, dependent of whether is free course, price and discount price
	 *
	 * @param string $post_id
	 * @param object $post
	 *
	 */
	function warren_core_course_save_final_price( $post_id, $post ) {

		if ( isset( $_POST ) && ! empty( $_POST ) ) {
			$free           = isset( $_POST['qodef_course_free'] ) ? $_POST['qodef_course_free'] : 'no';
			$price          = isset( $_POST['qodef_course_price'] ) ? $_POST['qodef_course_price'] : 0;
			$discount_price = isset( $_POST['qodef_course_discount_price'] ) ? $_POST['qodef_course_discount_price'] : '';

			if ( 'yes' === $free ) {
				update_post_meta( $post_id, 'qodef_course_final_price', 0 );

				return;
			}

			if ( ! empty( $discount_price ) ) {
				$final_price = $discount_price;
			} else {
				$final_price = $price;
			}

			update_post_meta( $post_id, 'qodef_course_final_price', intval( $final_price ) );
		}
	}

	add_action( 'save_post_course', 'warren_core_course_save_final_price', 5, 2 );
	add_action( 'edit_post_course', 'warren_core_course_save_final_price', 5, 2 );
}

if ( ! function_exists( 'warren_core_add_course_rating' ) ) {
	function warren_core_add_course_rating( $post_types ) {
		$post_types[] = 'course';

		return $post_types;
	}

	add_filter( 'warren_core_filter_rating_post_types', 'warren_core_add_course_rating' );
}

if ( ! function_exists( 'warren_core_get_course_status' ) ) {
	function warren_core_get_course_status( $id = '' ) {
		$id                 = ! empty( $id ) ? $id : get_the_ID();
		$free_course        = get_post_meta( $id, 'qodef_course_free', true );
		$prerequired_course = get_post_meta( $id, 'qodef_course_prerequired', true );
		$maximum_students   = get_post_meta( $id, 'qodef_course_maximum_students', true );
		$students_enrolled  = warren_core_get_course_students_enrolled_number( $id );

		if ( is_user_logged_in() ) {
			$user_id = get_current_user_id();

			$course_info = get_user_meta( $user_id, 'qodef_course_' . $id, true );

			//if course_info exists then user is enrolled in course
			if ( ! empty( $course_info ) ) {
				return 'enrolled';
			}

			if ( ! empty( $prerequired_course ) ) {
				$prerequired_course_info = get_user_meta( $user_id, 'qodef_course_' . $prerequired_course, true );

				if ( empty( $prerequired_course_info ) || 'yes' !== $prerequired_course_info['passed'] ) {
					return 'prerequired';
				}
			}

			if ( 'yes' === $free_course ) {
				return 'free';
			}

			$user_courses = warren_core_get_user_courses();

			if ( warren_core_check_is_course_in_cart( $id ) ) {
				return 'cart';
			}

			if ( in_array( $id, $user_courses['bought'] ) ) {
				return 'bought';
			} elseif ( in_array( $id, $user_courses['in_process'] ) ) {
				return 'process';
			}
		}

		if ( empty( $maximum_students ) || intval( $maximum_students ) > $students_enrolled ) {
			return 'buy';
		} else {
			return 'max_exceeded';
		}
	}
}

if ( ! function_exists( 'warren_core_check_is_course_in_cart' ) ) {
	function warren_core_check_is_course_in_cart( $id = '' ) {
		$id = ! empty( $id ) ? $id : get_the_ID();

		if ( null !== WC()->cart ) {
			foreach ( WC()->cart->get_cart() as $values ) {
				$_product = $values['data'];

				if ( $id === $_product->get_id() ) {
					return true;
				}
			}
		}

		return false;
	}
}

if ( ! function_exists( 'warren_core_get_course_students_enrolled_number' ) ) {
	function warren_core_get_course_students_enrolled_number( $id ) {
		$all_students           = get_post_meta( $id, 'qodef_course_members_repeater', true );
		$students_active_number = 0;

		if ( is_array( $all_students ) && count( $all_students ) ) {
			$students_active_number = count( $all_students );

			foreach ( $all_students as $key => $student ) {
				//if student status set to inactive, he is not counted
				if ( 'inactive' === $student['qodef_course_member_status'] ) {
					$students_active_number --;
				} else {
					$course_info = get_user_meta( $student['qodef_course_member_id'], 'qodef_course_' . $id, true );

					//if course passed or course max retakes used, this student has finished course
					if ( ! empty( $course_info ) && 'yes' === $course_info['passed'] ) {
						$students_active_number --;
					}
				}
			}
		}

		return $students_active_number;
	}
}

if ( ! function_exists( 'warren_core_get_course_popup' ) ) {
	/**
	 * Loads subscribe popup HTML
	 */
	function warren_core_get_course_popup() {
		if ( is_singular( 'course' ) ) {
			warren_core_template_part( 'plugins/lms/post-types/course', 'templates/popup' );
		}
	}

	// Get subscribe popup HTML
	add_action( 'warren_action_before_wrapper_close_tag', 'warren_core_get_course_popup' );
}

if ( ! function_exists( 'warren_core_course_get_prev_next' ) ) {
	function warren_core_course_get_prev_next( $course_id, $item_id ) {
		$prev_next_array     = array(
			'previous' => array(),
			'next'     => array(),
		);
		$all_items           = array();
		$curriculum_sections = get_post_meta( $course_id, 'qodef_course_curriculum_sections', true );

		$prev_index = false;
		$next_index = false;

		//arrange all subsection items into one array for easier search
		$all_items_index = 0;
		foreach ( $curriculum_sections as $section_key => $section ) {
			$subsection = $section['qodef_course_curriculum_subsections'];
			foreach ( $subsection as $key => $item ) {
				if ( 'lesson' === $item['qodef_course_subsection_item'] ) {
					$all_items[ $all_items_index ]['id']   = $item['qodef_course_subsection_item_lesson'];
					$all_items[ $all_items_index ]['type'] = 'lesson';
				} else {
					$all_items[ $all_items_index ]['id']   = $item['qodef_course_subsection_item_quiz'];
					$all_items[ $all_items_index ]['type'] = 'quiz';
				}
				$all_items_index ++;
			}
		}

		//go through all items in search for needed index
		foreach ( $all_items as $key => $item ) {
			if ( $item_id === $item['id'] ) {
				if ( 0 === $key && count( $all_items ) !== $key + 1 ) {
					$next_index = $key + 1;
				} elseif ( 0 !== $key && count( $all_items ) === $key + 1 ) {
					$prev_index = $key - 1;
				} elseif ( 0 !== $key && count( $all_items ) !== $key + 1 ) {
					$next_index = $key + 1;
					$prev_index = $key - 1;
				}

				break;
			}
		}

		if ( false !== $prev_index ) {
			$prev_next_array['previous']['id']    = $all_items[ $prev_index ]['id'];
			$prev_next_array['previous']['title'] = get_the_title( $all_items[ $prev_index ]['id'] );
			$prev_next_array['previous']['type']  = $all_items[ $prev_index ]['type'];
		}

		if ( false !== $next_index ) {
			$prev_next_array['next']['id']    = $all_items[ $next_index ]['id'];
			$prev_next_array['next']['title'] = get_the_title( $all_items[ $next_index ]['id'] );
			$prev_next_array['next']['type']  = $all_items[ $next_index ]['type'];
		}

		return $prev_next_array;
	}
}

if ( ! function_exists( 'warren_core_add_rest_api_course_global_variables' ) ) {
	/**
	 * Extend main rest api variables with new case
	 *
	 * @param array $global - list of variables
	 * @param string $namespace - rest namespace url
	 *
	 * @return array
	 */
	function warren_core_add_rest_api_course_global_variables( $global, $namespace ) {
		$global['getCourseEnrollRestRoute']  = $namespace . '/course-enroll';
		$global['getCourseContentRestRoute'] = $namespace . '/course-content';
		$global['getRetakeCourseRestRoute']  = $namespace . '/retake-course';

		return $global;
	}

	add_filter( 'warren_filter_rest_api_global_variables', 'warren_core_add_rest_api_course_global_variables', 10, 2 );
}

if ( ! function_exists( 'warren_core_add_rest_api_course_route' ) ) {
	/**
	 * Extend main rest api routes with new case
	 *
	 * @param array $routes - list of rest routes
	 *
	 * @return array
	 */
	function warren_core_add_rest_api_course_route( $routes ) {
		$routes['course-enroll']  = array(
			'route'               => 'course-enroll',
			'methods'             => WP_REST_Server::CREATABLE,
			'callback'            => 'warren_core_course_enroll',
			'permission_callback' => function () {
				return is_user_logged_in();
			},
			'args'                => array(
				'options' => array(
					'required'          => true,
					'validate_callback' => function ( $param, $request, $key ) {
						return is_array( $param ) ? $param : (array) $param;
					},
					'description'       => esc_html__( 'Options is required parameter', 'warren-core' ),
				),
			),
		);
		$routes['course-content'] = array(
			'route'    => 'course-content',
			'methods'  => WP_REST_Server::READABLE,
			'callback' => 'warren_core_get_course_content',
			'args'     => array(
				'options' => array(
					'required'          => true,
					'validate_callback' => function ( $param, $request, $key ) {
						return is_array( $param ) ? $param : (array) $param;
					},
					'description'       => esc_html__( 'Options is required parameter', 'warren-core' ),
				),
			),
		);
		$routes['retake-course']  = array(
			'route'               => 'retake-course',
			'methods'             => WP_REST_Server::CREATABLE,
			'callback'            => 'warren_core_retake_course',
			'permission_callback' => function () {
				return is_user_logged_in();
			},
			'args'                => array(
				'options' => array(
					'required'          => true,
					'validate_callback' => function ( $param, $request, $key ) {
						return is_array( $param ) ? $param : (array) $param;
					},
					'description'       => esc_html__( 'Options is required parameter', 'warren-core' ),
				),
			),
		);

		return $routes;
	}

	add_filter( 'warren_filter_rest_api_routes', 'warren_core_add_rest_api_course_route' );
}

if ( ! function_exists( 'warren_core_course_enroll' ) ) {
	function warren_core_course_enroll() {

		if ( ! isset( $_POST['options'] ) || empty( $_POST['options'] ) ) {
			qode_framework_get_ajax_status( 'error', esc_html__( 'Get method is invalid', 'warren-core' ) );
		} else {
			$options = array_filter( array_map( 'esc_attr', $_POST['options'] ) );

			$user_id            = get_current_user_id();
			$course_status      = warren_core_get_course_status( $options['course_id'] );
			$options['status']  = $course_status;
			$options['user_id'] = $user_id;

			if ( in_array( $course_status, array( 'free', 'bought' ), true ) ) {

				//enroll user in course, create meta fields
				$updated = warren_core_update_user_course_info( $user_id, $options['course_id'] );

				if ( $updated ) {
					$user_courses = get_user_meta( $user_id, 'qodef_courses_enrolled', true );

					if ( empty( $user_courses ) ) {
						$user_courses = array( $options['course_id'] );
					} else {
						if ( ! in_array( $user_id, $user_courses, true ) ) {
							$user_courses[] = $options['course_id'];
						}
					}

					update_user_meta( $user_id, 'qodef_courses_enrolled', $user_courses );

					$curriculum = warren_core_get_template_part( 'plugins/lms/post-types/course', 'templates/parts/post-info/curriculum-sections', '', $options );
					$progress   = warren_core_get_template_part( 'plugins/lms/post-types/course', 'templates/parts/post-info/status-progress', '', $options );

					$data = array(
						'curriculum' => $curriculum,
						'progress'   => $progress,
					);

					qode_framework_get_ajax_status( 'success', esc_html__( 'Course enrolled.', 'warren-core' ), $data );
				} else {
					qode_framework_get_ajax_status( 'error', esc_html__( 'Something went wrong, try again.', 'warren-core' ) );
				}
			} else {
				qode_framework_get_ajax_status( 'error', esc_html__( 'This course is not bought.', 'warren-core' ) );
			}
		}
	}
}
if ( ! function_exists( 'warren_core_get_course_content' ) ) {
	function warren_core_get_course_content() {

		if ( ! isset( $_GET['options'] ) || empty( $_GET['options'] ) ) {
			qode_framework_get_ajax_status( 'error', esc_html__( 'Get method is invalid', 'warren-core' ) );
		} else {
			$options = array_filter( array_map( 'esc_attr', $_GET['options'] ) );

			$free_lesson    = false;
			$user_logged_in = false;

			if ( 'lesson' === $options['item_type'] ) {
				$free_lesson = get_post_meta( $options['item_id'], 'qodef_lesson_free', true ) === 'yes' ? true : false;
			}

			if ( is_user_logged_in() ) {
				$user_logged_in = true;
			}

			if ( $user_logged_in || $free_lesson ) {
				$course_status     = warren_core_get_course_status( $options['course_id'] );
				$options['status'] = $course_status;

				if ( 'enrolled' === $course_status || $free_lesson ) {

					if ( 'lesson' === $options['item_type'] ) {
						$html = warren_core_get_template_part( 'plugins/lms/post-types/lesson', 'templates/content', '', $options );
					} else {
						$html = warren_core_get_template_part( 'plugins/lms/post-types/quiz', 'templates/content', '', $options );
					}

					$navigation = warren_core_get_template_part( 'plugins/lms/post-types/course', 'templates/navigation', '', $options );

					$data = array(
						'html'       => $html,
						'navigation' => $navigation,
					);

					qode_framework_get_ajax_status( 'success', esc_html__( 'Course opened.', 'warren-core' ), $data );
				} else {
					qode_framework_get_ajax_status( 'error', esc_html__( 'This course is not bought.', 'warren-core' ) );
				}
			} else {
				qode_framework_get_ajax_status( 'error', esc_html__( 'This is not allowed.', 'warren-core' ) );
			}
		}
	}
}

if ( ! function_exists( 'warren_core_retake_course' ) ) {
	function warren_core_retake_course() {

		if ( ! isset( $_POST['options'] ) || empty( $_POST['options'] ) ) {
			qode_framework_get_ajax_status( 'error', esc_html__( 'Get method is invalid', 'warren-core' ) );
		} else {
			$options = array_filter( array_map( 'esc_attr', $_POST['options'] ) );

			$user_id           = get_current_user_id();
			$course_status     = warren_core_get_course_status( $options['course_id'] );
			$options['status'] = $course_status;

			if ( 'enrolled' === $course_status ) {
				$course_info = get_user_meta( $user_id, 'qodef_course_' . $options['course_id'], true );

				warren_core_update_user_course_info( $user_id, $options['course_id'], array(), $course_info['retakes'] ++, $course_info['passed'] );

				qode_framework_get_ajax_status( 'success', esc_html__( 'Course Retaken', 'warren-core' ) );
			} else {
				qode_framework_get_ajax_status( 'error', esc_html__( 'This course is not bought.', 'warren-core' ) );
			}
		}
	}
}

if ( ! function_exists( 'warren_core_update_user_course_info' ) ) {
	function warren_core_update_user_course_info( $user_id, $course_id, $info = array(), $course_retakes = 0, $passed = 'no' ) {
		//if $info empty = just set initial values
		if ( empty( $info ) ) {
			$course_info_initial = array(
				'passed'     => $passed,
				'status'     => 'started',
				'percentage' => '0',
				'retakes'    => $course_retakes,
				'lessons'    => array(),
				'quizzes'    => array(),
			);

			$items = warren_core_course_items_ids( $course_id );

			foreach ( $items['lessons'] as $lesson ) {
				$course_info_initial['lessons'][ $lesson ] = array(
					'status' => 'to_start',
				);
			}

			foreach ( $items['quizzes'] as $quiz ) {
				$default_quiz_info = warren_core_get_default_quiz_options( $quiz, 0 );

				$course_info_initial['quizzes'][ $quiz ] = $default_quiz_info;
			}

			//if first time course taken, user should be added to course members list
			if ( 0 === $course_retakes ) {
				$members   = get_post_meta( $course_id, 'qodef_course_members_repeater', true );
				$user_data = get_userdata( $user_id );

				//if members not empty, add this user as a member or if he exists in array, change his status to enroll
				//if members empty, just add user
				if ( ! empty( $members ) ) {
					$exists = false;
					foreach ( $members as $key => $member ) {
						if ( $user_id == $member['qodef_course_member_id'] ) {
							$exists                                        = true;
							$members[ $key ]['qodef_course_member_status'] = 'enrolled';
							break;
						}
					}

					if ( ! $exists ) {
						$members[] = array(
							'qodef_course_member_id'     => $user_id,
							'qodef_course_member_name'   => $user_data->user_login,
							'qodef_course_member_email'  => $user_data->user_email,
							'qodef_course_member_status' => 'enrolled',
						);
					}
				} else {
					$members = array(
						array(
							'qodef_course_member_id'     => $user_id,
							'qodef_course_member_name'   => $user_data->user_login,
							'qodef_course_member_email'  => $user_data->user_email,
							'qodef_course_member_status' => 'bought',
						),
					);
				}
				update_post_meta( $course_id, 'qodef_course_members_repeater', $members );
			}

			return update_user_meta( $user_id, 'qodef_course_' . $course_id, $course_info_initial );
		} else {
			return update_user_meta( $user_id, 'qodef_course_' . $course_id, $info );
		}
	}
}

if ( ! function_exists( 'warren_core_course_items_ids' ) ) {
	/**
	 * Function that returns array of items ids in course
	 *
	 * @param string $course_id - id of the wanted course
	 *
	 * @return array $items
	 */
	function warren_core_course_items_ids( $course_id ) {
		$sections = get_post_meta( $course_id, 'qodef_course_curriculum_sections', true );
		$quizzes  = array();
		$lessons  = array();

		if ( count( $sections ) ) {
			foreach ( $sections as $section ) {
				$subsections = $section['qodef_course_curriculum_subsections'];
				if ( count( $subsections ) ) {
					foreach ( $subsections as $subsection ) {
						if ( 'quiz' === $subsection['qodef_course_subsection_item'] ) {
							$quizzes[] = $subsection['qodef_course_subsection_item_quiz'];
						} else {
							$lessons[] = $subsection['qodef_course_subsection_item_lesson'];
						}
					}
				}
			}
		}

		$items = array(
			'quizzes' => $quizzes,
			'lessons' => $lessons,
		);

		return $items;
	}
}

if ( ! function_exists( 'warren_core_get_item_status' ) ) {
	function warren_core_get_item_status( $course_id, $item_id, $item_type ) {
		if ( is_user_logged_in() ) {
			$course_status = get_user_meta( get_current_user_id(), 'qodef_course_' . $course_id, true );

			if ( ! empty( $course_status ) ) {
				return $course_status[ $item_type ][ $item_id ]['status'];
			} else {
				return false;
			}
		} else {
			return false;
		}
	}
}

if ( ! function_exists( 'warren_core_calculate_course_percentage' ) ) {
	/**
	 * Function that returns percentage of course calculated by previous percentage plus one for item currently completed
	 *
	 * @param array $course_info - info of course
	 *
	 * @return int
	 */
	function warren_core_calculate_course_percentage( $course_info ) {
		$all_items          = count( $course_info['lessons'] ) + count( $course_info['quizzes'] );
		$current_percentage = $course_info['percentage'];

		$items_completed = $current_percentage * $all_items / 100 + 1;

		return $items_completed * 100 / $all_items;
	}
}

if ( ! function_exists( 'warren_core_course_format_time' ) ) {
	function warren_core_course_format_time( $time ) {
		$time_unit   = substr( $time, - 1 );
		$time_number = intval( $time );
		$hours       = 0;
		$minutes     = 0;
		$seconds     = 0;

		switch ( $time_unit ) {
			case 's':
				$hours   = intval( $time_number / 3600 );
				$minutes = ( $time_number / 60 ) % 60;
				$seconds = $time_number % 60;
				break;
			case 'm':
				$hours   = intval( $time_number / 60 );
				$minutes = $time_number % 60;
				break;
			case 'h':
				$hours = $time_number;
				break;
		}

		if ( $hours < 10 ) {
			$hours = '0' . $hours;
		}

		if ( $minutes < 10 ) {
			$minutes = '0' . $minutes;
		}

		if ( $seconds < 10 ) {
			$seconds = '0' . $seconds;
		}

		return $hours . ':' . $minutes . ':' . $seconds;
	}
}
if ( ! function_exists( 'warren_core_get_course_title_text' ) ) {
	function warren_core_get_course_title_text( $title ) {

		if ( is_singular( 'course' ) ) {
			$title = '<span class="qodef-m-title-inner-holder">' . $title . '</span>' . warren_core_get_template_part( 'plugins/lms/post-types/course', 'templates/parts/post-info/price', 'alt' );
		}

		return $title;

	}

	add_filter( 'warren_filter_page_title_text', 'warren_core_get_course_title_text' );
}
