<?php

if ( ! function_exists( 'warren_core_enqueue_quiz_assets' ) ) {
	/**
	 * Function that enqueue 3rd party plugins script
	 */
	function warren_core_enqueue_quiz_assets() {

		if ( is_singular( 'course' ) ) {
			wp_enqueue_script( 'jquery-vtimer-min', WARREN_CORE_PLUGINS_URL_PATH . '/lms/post-types/quiz/assets/js/plugins/jquery.vtimer.min.js', array( 'jquery' ), false, true );
		}
	}

	add_action( 'warren_core_action_before_main_css', 'warren_core_enqueue_quiz_assets' );
}

if ( ! function_exists( 'warren_core_get_user_quiz_info' ) ) {
	function warren_core_get_user_quiz_info( $user_id, $course_id, $quiz_id ) {
		$quiz_info   = array(
			'passed'           => 'no',
			'status'           => 'to_start',
			'takes'            => 0,
			'current_question' => 0,
			'points'           => 0,
			'correct'          => 0,
			'incorrect'        => 0,
			'time_left'        => 0,
			'questions'        => array(),
		);
		$course_info = get_user_meta( $user_id, 'qodef_course_' . $course_id, true );

		if ( ! empty( $course_info ) && isset( $course_info['quizzes'][ $quiz_id ] ) ) {
			$quiz_info = $course_info['quizzes'][ $quiz_id ];
		}

		return $quiz_info;
	}
}

if ( ! function_exists( 'warren_core_add_rest_api_quiz_global_variables' ) ) {
	/**
	 * Extend main rest api variables with new case
	 *
	 * @param array $global - list of variables
	 * @param string $namespace - rest namespace url
	 *
	 * @return array
	 */
	function warren_core_add_rest_api_quiz_global_variables( $global, $namespace ) {
		$global['getQuizQuestionRestRoute'] = $namespace . '/quiz-question';
		$global['getQuizCanceledRestRoute'] = $namespace . '/quiz-cancel';
		$global['getQuizFinishedRestRoute'] = $namespace . '/quiz-finished';

		return $global;
	}

	add_filter( 'warren_filter_rest_api_global_variables', 'warren_core_add_rest_api_quiz_global_variables', 10, 2 );
}

if ( ! function_exists( 'warren_core_add_rest_api_quiz_route' ) ) {
	/**
	 * Extend main rest api routes with new case
	 *
	 * @param array $routes - list of rest routes
	 *
	 * @return array
	 */
	function warren_core_add_rest_api_quiz_route( $routes ) {
		$routes['quiz-question'] = array(
			'route'               => 'quiz-question',
			'methods'             => WP_REST_Server::CREATABLE,
			'callback'            => 'warren_core_get_quiz',
			'permission_callback' => function () {
				return is_user_logged_in();
			},
			'args'                => array(
				'options' => array(
					'required'          => true,
					'validate_callback' => function ( $param, $request, $key ) {
						return is_array( $param ) ? $param : (array) $param;
					},
					'description'       => esc_html__( 'Options is required parameter', 'warren-core' ),
				),
			),
		);
		$routes['quiz-cancel']   = array(
			'route'               => 'quiz-cancel',
			'methods'             => WP_REST_Server::CREATABLE,
			'callback'            => 'warren_core_cancel_quiz',
			'permission_callback' => function () {
				return is_user_logged_in();
			},
			'args'                => array(
				'options' => array(
					'required'          => true,
					'validate_callback' => function ( $param, $request, $key ) {
						return is_array( $param ) ? $param : (array) $param;
					},
					'description'       => esc_html__( 'Options is required parameter', 'warren-core' ),
				),
			),
		);
		$routes['quiz-finished'] = array(
			'route'               => 'quiz-finished',
			'methods'             => WP_REST_Server::CREATABLE,
			'callback'            => 'warren_core_finish_quiz',
			'permission_callback' => function () {
				return is_user_logged_in();
			},
			'args'                => array(
				'options' => array(
					'required'          => true,
					'validate_callback' => function ( $param, $request, $key ) {
						return is_array( $param ) ? $param : (array) $param;
					},
					'description'       => esc_html__( 'Options is required parameter', 'warren-core' ),
				),
			),
		);

		return $routes;
	}

	add_filter( 'warren_filter_rest_api_routes', 'warren_core_add_rest_api_quiz_route' );
}

if ( ! function_exists( 'warren_core_get_quiz' ) ) {
	function warren_core_get_quiz() {

		if ( ! isset( $_POST['options'] ) || empty( $_POST['options'] ) ) {
			qode_framework_get_ajax_status( 'error', esc_html__( 'Get method is invalid', 'warren-core' ) );
		} else {
			$options = array_filter( array_map( 'esc_attr', $_POST['options'] ) );
			$user_id = get_current_user_id();

			$course_status = warren_core_get_course_status( $options['course_id'] );

			if ( 'enrolled' === $course_status ) {

				$course_info = get_user_meta( $user_id, 'qodef_course_' . $options['course_id'], true );

				if ( isset( $options['retake'] ) && 'yes' === $options['retake'] ) {
					$course_info['quizzes'][ $options['quiz_id'] ] = warren_core_get_default_quiz_options( $options['quiz_id'], $course_info['quizzes'][ $options['quiz_id'] ]['takes'], $course_info['quizzes'][ $options['quiz_id'] ]['passed'] );
				}

				$course_info['quizzes'][ $options['quiz_id'] ]['status'] = 'started';

				warren_core_update_user_course_info( $user_id, $options['course_id'], $course_info );

				$html = warren_core_get_template_part( 'plugins/lms/post-types/question', 'templates/content', '', $options );

				$data = array(
					'html' => $html,
				);

				qode_framework_get_ajax_status( 'success', esc_html__( 'Quiz started.', 'warren-core' ), $data );
			} else {
				qode_framework_get_ajax_status( 'error', esc_html__( 'You must buy a course', 'warren-core' ) );
			}
		}
	}
}

if ( ! function_exists( 'warren_core_cancel_quiz' ) ) {
	function warren_core_cancel_quiz() {

		if ( ! isset( $_POST['options'] ) || empty( $_POST['options'] ) ) {
			qode_framework_get_ajax_status( 'error', esc_html__( 'Get method is invalid', 'warren-core' ) );
		} else {
			$options = array_filter( array_map( 'esc_attr', $_POST['options'] ) );
			$user_id = get_current_user_id();

			$course_status = warren_core_get_course_status( $options['course_id'] );

			if ( 'enrolled' === $course_status ) {

				$course_info = get_user_meta( $user_id, 'qodef_course_' . $options['course_id'], true );

				$course_info['quizzes'][ $options['quiz_id'] ]['time_left'] = $options['time_left'];

				if ( isset( $options['answers'] ) && ! empty( $options['answers'] ) ) {
					$course_info['quizzes'][ $options['quiz_id'] ]['questions'][ $options['question_id'] ]['current_answer'] = $options['answers'];
				}

				warren_core_update_user_course_info( $user_id, $options['course_id'], $course_info );

				qode_framework_get_ajax_status( 'success', esc_html__( 'Quiz paused.', 'warren-core' ) );
			} else {
				qode_framework_get_ajax_status( 'error', esc_html__( 'You must buy a course', 'warren-core' ) );
			}
		}
	}
}

if ( ! function_exists( 'warren_core_finish_quiz' ) ) {
	function warren_core_finish_quiz() {

		if ( ! isset( $_POST['options'] ) || empty( $_POST['options'] ) ) {
			qode_framework_get_ajax_status( 'error', esc_html__( 'Get method is invalid', 'warren-core' ) );
		} else {
			$options = array_filter( array_map( 'esc_attr', $_POST['options'] ) );
			$user_id = get_current_user_id();

			$course_status = warren_core_get_course_status( $options['course_id'] );

			if ( 'enrolled' === $course_status ) {

				$course_info = get_user_meta( $user_id, 'qodef_course_' . $options['course_id'], true );

				if ( isset( $options['answers'] ) && ! empty( $options['answers'] ) ) {
					$course_info['quizzes'][ $options['quiz_id'] ]['questions'][ $options['question_id'] ]['current_answer'] = $options['answers'];
				}

				$time = get_post_meta( $options['quiz_id'], 'qodef_quiz_duration', true );
				$unit = get_post_meta( $options['quiz_id'], 'qodef_quiz_duration_unit', true );

				switch ( $unit ) {
					case 'seconds':
						$unit = 's';
						break;
					case 'hours':
						$unit = 'h';
						break;
					default:
						$unit = 'm';
						break;
				}

				$time .= $unit;

				$quiz_stats = warren_core_calculate_quiz_stats( $course_info['quizzes'][ $options['quiz_id'] ] );

				$course_info['quizzes'][ $options['quiz_id'] ]['status']           = 'finished';
				$course_info['quizzes'][ $options['quiz_id'] ]['takes']           += 1;
				$course_info['quizzes'][ $options['quiz_id'] ]['current_question'] = 0;
				$course_info['quizzes'][ $options['quiz_id'] ]['time_left']        = $time;
				$course_info['quizzes'][ $options['quiz_id'] ]['points']           = $quiz_stats['points'];
				$course_info['quizzes'][ $options['quiz_id'] ]['correct']          = $quiz_stats['correct'];
				$course_info['quizzes'][ $options['quiz_id'] ]['incorrect']        = $quiz_stats['incorrect'];

				//if user has not passed quiz, check if he had passed now (if user already passed the quiz once, do not check again)
				if ( 'no' === $course_info['quizzes'][ $options['quiz_id'] ]['passed'] ) {
					$passing_percent = intval( get_post_meta( $options['quiz_id'], 'qodef_quiz_passing_percentage', true ) );
					$percentage      = $quiz_stats['points'] / $course_info['quizzes'][ $options['quiz_id'] ]['total_points'] * 100;

					//if user has passed quiz, add passed flag and add percentage to the course
					if ( $percentage >= $passing_percent ) {
						$course_info['quizzes'][ $options['quiz_id'] ]['passed'] = 'yes';

						$new_percentage            = warren_core_calculate_course_percentage( $course_info );
						$course_info['percentage'] = $new_percentage;
						$course_passing_percentage = get_post_meta( $options['course_id'], 'qodef_course_passing_percentage', true );

						if ( 100 === intval( $new_percentage ) ) {
							$course_info['status'] = 'completed';
						}
						if ( intval( $course_passing_percentage ) <= intval( $new_percentage ) ) {
							$course_info['passed'] = 'yes';
						}
					}
				}

				warren_core_update_user_course_info( $user_id, $options['course_id'], $course_info );

				//set params to be passed to stats, necessary only quiz params and course_id
				$params = array(
					'user_id'   => $user_id,
					'course_id' => $options['course_id'],
					'quiz_id'   => $options['quiz_id'],
				);
				$params = array_merge( $params, $course_info['quizzes'][ $options['quiz_id'] ] );

				$html     = warren_core_get_template_part( 'plugins/lms/post-types/quiz', 'templates/stats', '', $params );
				$progress = warren_core_get_template_part( 'plugins/lms/post-types/course', 'templates/parts/post-info/status-progress', '', $params );

				$data = array(
					'html'       => $html,
					'time'       => $time,
					'percentage' => $course_info['percentage'],
					'progress'   => $progress,
				);

				qode_framework_get_ajax_status( 'success', esc_html__( 'Quiz finished.', 'warren-core' ), $data );
			} else {
				qode_framework_get_ajax_status( 'error', esc_html__( 'You must buy a course', 'warren-core' ) );
			}
		}
	}
}

if ( ! function_exists( 'warren_core_calculate_quiz_stats' ) ) {
	function warren_core_calculate_quiz_stats( $quiz_info ) {
		$stats = array(
			'points'    => 0,
			'correct'   => 0,
			'incorrect' => 0,
		);

		foreach ( $quiz_info['questions'] as $question_id => $question ) {
			$correct_answer = warren_core_question_get_answer( $question_id );
			$current_answer = explode( '()', $question['current_answer'] );

			if ( ! is_array( $current_answer ) ) {
				$current_answer = array( $current_answer );
			}

			//check if contents of two arrays are identical, needs to be == because === checks the order as well
			if ( $correct_answer == $current_answer ) {
				$stats['correct'] += 1;

				$points           = intval( get_post_meta( $question_id, 'qodef_question_points', true ) );
				$stats['points'] += $points;
			} else {
				$stats['incorrect'] += 1;
			}
		}

		return $stats;
	}
}

if ( ! function_exists( 'warren_core_get_default_quiz_options' ) ) {
	function warren_core_get_default_quiz_options( $quiz_id, $takes = 0, $passed = 'no' ) {
		$time = get_post_meta( $quiz_id, 'qodef_quiz_duration', true );
		$unit = get_post_meta( $quiz_id, 'qodef_quiz_duration_unit', true );

		switch ( $unit ) {
			case 'seconds':
				$unit = 's';
				break;
			case 'hours':
				$unit = 'h';
				break;
			default:
				$unit = 'm';
				break;
		}

		$time .= $unit;

		$questions     = array();
		$questions_all = get_post_meta( $quiz_id, 'qodef_quiz_questions', true );
		$total_points  = 0;

		foreach ( $questions_all as $question ) {
			$questions[ $question['qodef_quiz_question'] ] = array(
				'current_answer' => '',
				'checked'        => 'no',
			);

			$total_points += intval( get_post_meta( $question['qodef_quiz_question'], 'qodef_question_points', true ) );
		}

		$default_info = array(
			'passed'           => $passed,
			'status'           => 'to_start',
			'takes'            => $takes,
			'current_question' => 0,
			'total_points'     => $total_points,
			'points'           => 0,
			'correct'          => 0,
			'incorrect'        => 0,
			'time_left'        => $time,
			'questions'        => $questions,
		);

		return $default_info;
	}
}
